"""
Author: lidianycs

Description: 
This script uses Selenium to scrape articles from a DEV page. It specifically extracts the title and URL from <article> tags 
with a class "crayons-story" and saves the results to a CSV file.

Prerequisites:
1. Python installed on your system.
2. Install the Selenium package:
   pip install selenium
3. Download and set up the WebDriver for your browser:
   - ChromeDriver: https://sites.google.com/chromium.org/driver/
   - GeckoDriver for Firefox: https://github.com/mozilla/geckodriver/releases
   Make sure the WebDriver is in your system's PATH or specify its full path in the script.

How to Run:
1. Replace the 'base_url' variable with the URL of the webpage you want to scrape.
2. Specify the desired output CSV file name in the 'output_csv' variable.
3. Run the script in your Python environment:
   python script_name.py
4. The script will open a browser, fetch the webpage, and save the extracted data to the specified CSV file.

License:
This file is licensed under the **Creative Commons Attribution-ShareAlike 4.0 International License (CC BY-SA 4.0)**.

You are free to share, adapt, and build upon the materials in this package, as long as you provide proper attribution to the original work and share any derivative works under the same license.

For more details, please refer to the [CC BY-SA 4.0 License](https://creativecommons.org/licenses/by-sa/4.0/).
"""


from selenium import webdriver
from selenium.webdriver.common.by import By
import csv
from urllib.parse import urljoin  # To build full URLs
import time  # For adding delays if needed

def scrape_articles(base_url, output_csv):
    # Configure the WebDriver (Chrome in this example)
    driver = webdriver.Chrome()  # Use your preferred browser's driver
    driver.get(base_url)  # Open the webpage
    
    time.sleep(5)  # Wait for the page to load fully (adjust as needed)
    
    # Find all <article> tags with the class 'crayons-story'
    articles = driver.find_elements(By.CLASS_NAME, "crayons-story")
    
    # List to store the scraped data
    scraped_data = []
    
    # Extract the title and URL from each article
    for article in articles:
        try:
            # Locate the link inside the article
            link = article.find_element(By.CLASS_NAME, "crayons-story__hidden-navigation-link")
            title = link.text  # Extract the text (title)
            relative_url = link.get_attribute("href")  # Extract the full URL
            scraped_data.append({'Title': title, 'URL': relative_url})
        except Exception as e:
            print(f"Error extracting data for an article: {e}")
    
    # Save the data to a CSV file
    with open(output_csv, mode='w', newline='', encoding='utf-8') as csv_file:
        writer = csv.DictWriter(csv_file, fieldnames=['Title', 'URL'])
        writer.writeheader()
        writer.writerows(scraped_data)
    
    print(f"Scraped {len(scraped_data)} articles. Data saved to {output_csv}.")
    
    # Close the browser
    driver.quit()

# Replace this with the target URL
base_url = "https://dev.to/search?utf8=%E2%9C%93&q=empathy"
output_csv = "scraped_articles.csv"
scrape_articles(base_url, output_csv)
